﻿#region Test support members
#region Constants

/// <summary>
/// Строка в сообщении валидации соответствующая успешному выполнению теста.
/// </summary>
private const string PassedStr = "Passed";

/// <summary>
/// Строка в сообщении валидации соответствующая ошибке при выполнении теста.
/// </summary>
private const string FailedStr = "Failed";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится значение флага, показывающего, запущен процесс из тестов или нет. Значение типа: <see cref="bool"/>.
/// </summary>
private const string IsLaunchedInTestKey = "IsLaunchedInTest";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится метод инициализации бизнес-процесса при выполнении из тестов. Значение типа: <see cref="Func{T, TResult}"/>, где T - <see cref="WorkflowEngineCompiledBase"/>, TResult - <see cref="ValueTask"/>.
/// </summary>
private const string TestInitializerActionKey = "TestInitializerAction";

#endregion

#region Properties

/// <summary>
/// Возвращает значение, показывающее, что процесс запущен из тестов.
/// </summary>
protected bool IsLaunchedInTest => this.ProcessHash.TryGet<bool>(IsLaunchedInTestKey);

#endregion

#region Protected Methods

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком успешного выполнения теста "<see cref="PassedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="PassedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Passed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    var str = PassedStr;

    if (!string.IsNullOrEmpty(suffix))
    {
        str += " " + suffix;
    }

    this.AddInfo(str);
}

/// <summary>
/// Добавляет в результаты валидации сообщение типа <see cref="ValidationResultType.Error"/> с сообщением об успешном выполнении.
/// </summary>
/// <remarks>
/// Созданное сообщение предназначено для остановки выполнения бизнес-процесса. Для этого при проверке результатов выполнения необходимо в методе <see cref="T:Tessa.Test.Default.Shared.Workflow.WeAssert.Passed"/> разрешить наличие ошибок в результате валидации.<para/>
/// Не выполняет действий, если процесс выполняется не из тестов.
/// </remarks>
protected void PassedWithStop()
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(PassedStr);
}

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком ошибки в тесте "<see cref="FailedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="FailedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Failed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(FailedStr + " " + suffix);
}

/// <summary>
/// Инициализирует бизнес-процесс при выполнении из тестов.
/// </summary>
/// <returns>Асинхронная задача.</returns>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected async ValueTask InitProcessAsync()
{
    var card = this.StoreCardObject;

    if(card is null)
    {
        return;
    }

    var info = card.TryGetInfo();
    
    if (info != null
        && info.Remove(IsLaunchedInTestKey, out var isLaunchedInTestObj)
        && ((bool) isLaunchedInTestObj))
    {
        this.ProcessHash.Add(IsLaunchedInTestKey, BooleanBoxes.True);

        if (info.Remove(TestInitializerActionKey, out object testInitializerFuncAsyncObj))
        {
            if(testInitializerFuncAsyncObj != null)
            {
                var initFuncAsync = (Func<WorkflowEngineCompiledBase, ValueTask>) testInitializerFuncAsyncObj;
                await initFuncAsync(this);
            }
        }
    }
}

#endregion
#endregion

protected void AddReworkComent(CardTask task)
{
	var section = task.Card.Sections.GetOrAdd("WfResolutions");
	string comment = section.RawFields.TryGet<string>("Comment");
	
	Process.ReworkComment = (string)Process.ReworkComment +
		(!string.IsNullOrEmpty(comment)
		? string.Format("{0} ({1}): {2}\n", task.UserName , DateTime.Now, comment)
		: string.Format("{0} ({1})\n", task.UserName , DateTime.Now));

	Process.NeedRework = true;
}

protected void InitApprovement()
{
	Process.ReworkComment = null;
	Process.NeedRework = false;
}
